#!/bin/bash
echo The Inobitec DICOM Server \(PACS\) installation script

if [ "$(id -u)" != "0" ]; then
  echo "This script must be run as root" 1>&2
  exit 1
fi

inst_dir=/srv/PACSServer
if [ -d "$inst_dir" ]; then
  echo "The directory ${inst_dir} is already exists."
  echo "Delete the folder or use update_pacs.sh script to upgrade"
  echo "an existing version of Inobitec DICOM Server (PACS)."
  exit 1
fi

echo Dependency check...
cd ../bin || exit 1
missed_dependencies_count="$(ldd ../bin/pacs_server | grep -v "libcharset" | grep -v "libiconv" | grep -c "not found")"
if [ "$missed_dependencies_count" -gt 0 ]; then
  echo "Follow components are requered for Inobitec DICOM Server (PACS) and absent:"
  ldd ../bin/pacs_server | grep -v "libcharset" | grep -v "libiconv" | grep "not found"
  echo "Install the dependencies and try to install Inobitec DICOM Server (PACS) again."
  exit 1
else
  echo "All dependencies are satisfied."
fi
cd ../Scripts || exit 1

function valid_ip()
{
  local ip=$1
  local stat=1
  if [[ $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
    OIFS=$IFS
    IFS='.'
    ip=($ip)
    IFS=$OIFS
    [[ ${ip[0]} -le 255 && ${ip[1]} -le 255 && ${ip[2]} -le 255 && ${ip[3]} -le 255 ]]
    stat=$?
  fi

  return $stat
}

while true; do
  read -rp "Enter database host ip [127.0.0.1]: " db_host
  db_host=${db_host:-127.0.0.1}
  if ! valid_ip "$db_host"; then
    echo "Wrong ipv4 address format."
  else
    break
  fi
done

re='^[0-9]+$'

while true; do
  read -rp "Enter database port [5432]: " db_port
  db_port=${db_port:-5432}
  if ! [[ $db_port =~ $re ]]; then
    echo "The port must contains only the numbers."
  else
    break
  fi
done

read -rp "Enter database username [PACS_USER]: " db_user
db_user=${db_user:-PACS_USER}

read -rsp "Enter database user password: " db_pass
echo ""

while true; do
  read -rp "Deploy database on thе ${db_host} machine? [y/n] " yn
  case $yn in
    [Yy]* ) deploy_database_here=true; break;;
    [Nn]* ) deploy_database_here=false; break;;
    * ) echo "Please answer \"y\" or \"n\".";;
  esac
done
if [ "$deploy_database_here" == "true" ]; then
  read -rp "Enter database admin username [postgres]: " db_admin_user
  db_admin_user=${db_admin_user:-postgres}

  while true; do
    read -rsp "Enter database admin password: " db_admin_pass
  if [ -z "${db_admin_pass}" ]; then
    echo -e "\nThe password can not be empty."
  else
    break;
  fi
  done
  echo ""
fi


read -rp "Enter the Inobitec DICOM server AETitle [PACS_Inobitec]: " pacs_server_ae_title
pacs_server_ae_title=${pacs_server_ae_title:-PACS_Inobitec}

while true; do
  read -rp "Accept associations from any AETitles? [y/n] " yn
  case $yn in
    [Yy]* ) accept_association_from_any=true; break;;
    [Nn]* ) accept_association_from_any=false; break;;
    * ) echo "Please answer \"y\" or \"n\".";;
  esac
done

while true; do
  read -rp "Activate unsecured DICOM store service? [y/n] " yn
  case $yn in
    [Yy]* ) activate_unsecure_dicom_store_service=true; break;;
    [Nn]* ) activate_unsecure_dicom_store_service=false; break;;
    * ) echo "Please answer \"y\" or \"n\".";;
  esac
done
if [ "$activate_unsecure_dicom_store_service" == "true" ]; then
  while true; do
    read -rp "Enter the unsecured DICOM store service port [3000]: " unsecure_dicom_store_service_port
    unsecure_dicom_store_service_port=${unsecure_dicom_store_service_port:-3000}
    if ! [[ $unsecure_dicom_store_service_port =~ $re ]]; then
      echo "This value must contains only the numbers."
    else
      break
    fi
  done
else
  unsecure_dicom_store_service_port=3000
fi

while true; do
  read -rp "Activate unsecured DICOM MWL service? [y/n] " yn
  case $yn in
    [Yy]* ) activate_unsecure_dicom_mwl_service=true; break;;
    [Nn]* ) activate_unsecure_dicom_mwl_service=false; break;;
    * ) echo "Please answer \"y\" or \"n\".";;
  esac
done
if [ "$activate_unsecure_dicom_mwl_service" == "true" ]; then
  while true; do
    read -rp "Enter the unsecured DICOM MWL service port [3001]: " unsecure_dicom_mwl_service_port
    unsecure_dicom_mwl_service_port=${unsecure_dicom_mwl_service_port:-3001}
    if ! [[ $unsecure_dicom_mwl_service_port =~ $re ]]; then
      echo "This value must contains only the numbers."
    else
      break
    fi
  done
else
  unsecure_dicom_mwl_service_port=3001
fi

while true; do
  read -rp "Activate secured DICOM store service? [y/n] " yn
  case $yn in
    [Yy]* ) activate_secure_dicom_store_service=true; break;;
    [Nn]* ) activate_secure_dicom_store_service=false; break;;
    * ) echo "Please answer \"y\" or \"n\".";;
  esac
done
if [ "$activate_secure_dicom_store_service" == "true" ]; then
  while true; do
    read -rp "Enter the secured DICOM store service port [3100]: " secure_dicom_store_service_port
    secure_dicom_store_service_port=${secure_dicom_store_service_port:-3100}
    if ! [[ $secure_dicom_store_service_port =~ $re ]]; then
      echo "This value must contains only the numbers."
    else
      break
    fi
  done
else
  secure_dicom_store_service_port=3100
fi

while true; do
  read -rp "Activate secured DICOM MWL service? [y/n] " yn
  case $yn in
    [Yy]* ) activate_secure_dicom_mwl_service=true; break;;
    [Nn]* ) activate_secure_dicom_mwl_service=false; break;;
    * ) echo "Please answer \"y\" or \"n\".";;
  esac
done
if [ "$activate_secure_dicom_mwl_service" == "true" ]; then
  while true; do
    read -rp "Enter the secured DICOM MWL service port [3101]: " secure_dicom_mwl_service_port
    secure_dicom_mwl_service_port=${secure_dicom_mwl_service_port:-3101}
    if ! [[ $secure_dicom_mwl_service_port =~ $re ]]; then
      echo "This value must contains only the numbers."
    else
      break
    fi
  done
else
  secure_dicom_mwl_service_port=3101
fi

while true; do
  read -rp "Enter free space storage limit [2048]: " free_disk_space_limit
  free_disk_space_limit=${free_disk_space_limit:-2048}
  if ! [[ $free_disk_space_limit =~ $re ]]; then
    echo "This value must contains only the numbers."
  else
    break
  fi
done

while true; do
  read -rp "Enter the HTTP service port [8000]: " unsecure_http_service_port
  unsecure_http_service_port=${unsecure_http_service_port:-8000}
  if ! [[ $unsecure_http_service_port =~ $re ]]; then
    echo "This value must contains only the numbers."
  else
    break
  fi
done

read -rp "Enter the WebAdmin user name [Admin]: " web_admin_login
web_admin_login=${web_admin_login:-Admin}

while true; do
  read -rsp "Enter the WebAdmin user password: " web_admin_password
  if [ -z "${web_admin_password}" ]; then
    echo -e "\nThe password can not be empty."
  else
    break;
  fi
done
echo -e ""

read -rp "Enter the Inobitec DICOM Server (PACS) service name [pacsserver]: " service_name
service_name=${service_name:-pacsserver}
read -rp "Enter the user name for starting Inobitec DICOM Server (PACS) service [pacs]: " service_user_name
service_user_name=${service_user_name:-pacs}

read -rp "Enter the DBMS service name (leave blank if the database is not deployed on this computer): " db_service_name

while true; do
  read -rp "Enter the amount of free storage space at which a notification will be sent [4096]: " free_disk_space_warning_notify_limit
  free_disk_space_warning_notify_limit=${free_disk_space_warning_notify_limit:-4096}
  if ! [[ $free_disk_space_warning_notify_limit =~ $re ]]; then
    echo "This value must contains only the numbers."
  else
    break
  fi
done

script_dir=${PWD}
config_dir=/etc/pacs-srv
log_dir=/var/log/pacs-srv
echo -e "\nCopying files..."
mkdir ${inst_dir} || { echo "Copy the files failed."; exit 1; }
mkdir ${inst_dir}/bin || { echo "Copy the files failed."; exit 1; }
mkdir ${inst_dir}/Fonts || { echo "Copy the files failed."; exit 1; }
mkdir ${inst_dir}/certs || { echo "Copy the files failed."; exit 1; }
mkdir ${inst_dir}/certs/trusted || { echo "Copy the files failed."; exit 1; }
mkdir ${config_dir} || { echo "Copy the files failed."; exit 1; }
mkdir ${log_dir} || { echo "Copy the files failed."; exit 1; }

cp ../bin/pacs_server ${inst_dir}/bin || { echo "Copy the files failed."; exit 1; }
cp -d ../bin/lib* ${inst_dir}/bin || { echo "Copy the files failed."; exit 1; }
cp ../Fonts/* ${inst_dir}/Fonts || { echo "Copy the files failed."; exit 1; }
cp -R ../Certs/* ${inst_dir}/certs || { echo "Copy the files failed."; exit 1; }
cd ${inst_dir} || { echo "Copy the files failed."; exit 1; }
ln -s ${config_dir} conf || { echo "Copy the files failed."; exit 1; }
ln -s ${log_dir} logs || { echo "Copy the files failed."; exit 1; }

echo Creating configuration file...
# srv_settings.ini
srv_config_file=${config_dir}/srv_settings.ini
{
  echo "database/conn_string=Server=\"${db_host};\"Port=\"${db_port};\"Database=\"pacs;\"unix_socket=\"NULL;\"client_flag=\"0;\""
  echo "database/user=${db_user}"
  echo "database/pass=${db_pass}"
  echo "database/driver=QPSQL"
  echo "database/max_connections_number=100"
  echo "http_server/port=${unsecure_http_service_port}"
  echo "web_admin/font_path=${inst_dir}/Fonts/font.ttf"
  echo "dicom/store_service/unsecure/enabled=${activate_unsecure_dicom_store_service}"
  echo "dicom/store_service/unsecure/port=${unsecure_dicom_store_service_port}"
  echo "dicom/accept_unknown=${accept_association_from_any}"
  echo "dicom/ae_title=${pacs_server_ae_title}"
  echo "storage/free_disk_space_limit=${free_disk_space_limit}"
  echo "dicom/mwl_service/unsecure/enabled=${activate_unsecure_dicom_mwl_service}"
  echo "dicom/mwl_service/unsecure/port=${unsecure_dicom_mwl_service_port}"
  echo "dicom/store_service/secure/enabled=${activate_secure_dicom_store_service}"
  echo "dicom/store_service/secure/port=${secure_dicom_store_service_port}"
  echo "dicom/mwl_service/secure/enabled=${activate_secure_dicom_mwl_service}"
  echo "dicom/mwl_service/secure/port=${secure_dicom_mwl_service_port}"
  echo "web_admin/login=${web_admin_login}"
  echo "web_admin/password=${web_admin_password}"
  echo "web_admin/session_timeout=1800"
  echo "dicom/tls/scu_private_key=${inst_dir}/certs/PACS_Client_private.key"
  echo "dicom/tls/scu_public_cert=${inst_dir}/certs/PACS_Client_public.crt"
  echo "dicom/tls/scp_private_key=${inst_dir}/certs/PACS_Server_private.key"
  echo "dicom/tls/scp_public_cert=${inst_dir}/certs/PACS_Server_public.crt"
  echo "dicom/tls/trusted_certs_path=${inst_dir}/certs/trusted"
  echo "compression/level=0"
  echo "compression/compress_after=365"
  echo "autoremove/store_days=0"
  echo "regional_pacs/enabled=false"
  echo "notifications/storage_free_space_limit=${free_disk_space_warning_notify_limit}"
  echo "logger/files_path=${log_dir}"
} > "${srv_config_file}"

if ! id -u "${service_user_name}" > /dev/null 2>&1; then
  echo Creating user...
  useradd -m -s /bin/bash "${service_user_name}" || { echo "Create user failed."; exit 1; }
fi

chown -R "${service_user_name}": ${config_dir} || { echo "Set owner for directories failed."; exit 1; }
chown -R "${service_user_name}": ${log_dir} || { echo "Set owner for directories failed."; exit 1; }
chown -R "${service_user_name}": ${inst_dir}/certs || { echo "Set owner for directories failed."; exit 1; }

if [ "${deploy_database_here}" != "false" ]; then
  echo "Deploying database..."
  cd ${script_dir}
  echo '' | ./sql_manager deploy ${db_host} ${db_port} ${db_admin_user} ${db_admin_pass} ${db_user} ${db_pass}
  cd ${inst_dir}
fi

echo Installing service...
service_file=/etc/systemd/system/${service_name}.service
{
  echo "[Unit]"
  echo "Description=Inobitec DICOM Server (PACS)"
  echo "After=network.target ${db_service_name} ${render_node_service_name}"
  echo ""
  echo "[Install]"
  echo "WantedBy=multi-user.target"
  echo ""
  echo "[Service]"
  echo "Type=simple"
  echo "User=${service_user_name}"
  echo "Group=${service_user_name}"
  echo "PermissionsStartOnly=true"
  echo "ExecStart=${inst_dir}/bin/pacs_server -cfg ${inst_dir}/conf/srv_settings.ini"
  echo "TimeoutSec=300"
  echo "WorkingDirectory=${inst_dir}/bin"
} > "${service_file}"
systemctl daemon-reload || { echo "Install the service failed. Check the ${service_file} and try \"# systemctl daemon-reload\" again."; exit 1; }
systemctl enable "${service_name}" || { echo "Failed to enable ${service_name}."; exit 1; }

echo Installing Inobitec DICOM Server \(PACS\) completed.
